import React, { createContext, useContext, useState, ReactNode, useEffect } from 'react';

export type ThemeMode = 'light' | 'dusky';
export type ThemeVariant = 'floral' | 'simple' | 'cabin' | 'nightSky';

interface ThemeContextType {
  currentMode: ThemeMode;
  currentVariant: ThemeVariant;
  setCurrentMode: (mode: ThemeMode) => void;
  setCurrentVariant: (variant: ThemeVariant) => void;
  toggleMode: () => void;
}

// Helper function to validate theme variant
const isValidThemeVariant = (variant: string | null): variant is ThemeVariant => {
  return variant === 'floral' || variant === 'simple' || variant === 'cabin' || variant === 'nightSky';
};

// Helper function to validate theme mode
const isValidThemeMode = (mode: string | null): mode is ThemeMode => {
  return mode === 'light' || mode === 'dusky';
};

const ThemeContext = createContext<ThemeContextType | undefined>(undefined);

export function ThemeProvider({ children }: { children: ReactNode }) {
  // Initialize state from localStorage or default to 'light'
  const [currentMode, setCurrentMode] = useState<ThemeMode>(() => {
    try {
      const savedMode = localStorage.getItem('themeMode');
      return isValidThemeMode(savedMode) ? savedMode : 'light';
    } catch (e) {
      return 'light';
    }
  });
  
  const [currentVariant, setCurrentVariant] = useState<ThemeVariant>(() => {
    try {
      const savedVariant = localStorage.getItem('themeVariant');
      return isValidThemeVariant(savedVariant) ? savedVariant : 'floral';
    } catch (e) {
      return 'floral';
    }
  });

  // Custom setter that updates localStorage
  const setThemeMode = (mode: ThemeMode) => {
    setCurrentMode(mode);
    try {
      localStorage.setItem('themeMode', mode);
    } catch (e) {
      console.warn('Failed to save theme mode to localStorage:', e);
    }
  };

  // Custom setter that updates localStorage
  const setThemeVariant = (variant: ThemeVariant) => {
    setCurrentVariant(variant);
    try {
      localStorage.setItem('themeVariant', variant);
    } catch (e) {
      console.warn('Failed to save theme variant to localStorage:', e);
    }
  };

  const toggleMode = () => {
    const newMode = currentMode === 'light' ? 'dusky' : 'light';
    setThemeMode(newMode);
  };

  // Add theme class to document root for global CSS access
  useEffect(() => {
    document.documentElement.setAttribute('data-theme', currentMode);
  }, [currentMode]);

  return (
    <ThemeContext.Provider 
      value={{ 
        currentMode, 
        currentVariant, 
        setCurrentMode: setThemeMode, 
        setCurrentVariant: setThemeVariant,
        toggleMode 
      }}
    >
      {children}
    </ThemeContext.Provider>
  );
}

export function useTheme() {
  const context = useContext(ThemeContext);
  if (context === undefined) {
    throw new Error('useTheme must be used within a ThemeProvider');
  }
  return context;
}

export default ThemeContext;